package nemosofts.voxradio.activity;

import static android.Manifest.permission.READ_EXTERNAL_STORAGE;
import static android.Manifest.permission.READ_MEDIA_AUDIO;
import static android.Manifest.permission.WRITE_EXTERNAL_STORAGE;

import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.FrameLayout;
import android.widget.Toast;

import androidx.activity.OnBackPressedCallback;
import androidx.activity.result.ActivityResultLauncher;
import androidx.activity.result.contract.ActivityResultContracts;
import androidx.annotation.NonNull;
import androidx.annotation.OptIn;
import androidx.core.content.ContextCompat;
import androidx.core.view.GravityCompat;
import androidx.drawerlayout.widget.DrawerLayout;
import androidx.fragment.app.Fragment;
import androidx.fragment.app.FragmentManager;
import androidx.fragment.app.FragmentTransaction;
import androidx.media3.common.util.UnstableApi;
import androidx.nemosofts.material.ToggleView;
import androidx.nemosofts.utils.NetworkUtils;

import com.google.android.gms.tasks.Task;
import com.google.android.material.navigation.NavigationView;
import com.google.android.play.core.review.ReviewInfo;
import com.google.android.play.core.review.ReviewManager;
import com.google.android.play.core.review.ReviewManagerFactory;
import com.onesignal.Continue;
import com.onesignal.OneSignal;
import com.sothree.slidinguppanel.SlidingUpPanelLayout;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import nemosofts.voxradio.R;
import nemosofts.voxradio.callback.Callback;
import nemosofts.voxradio.dialog.DialogUtil;
import nemosofts.voxradio.executor.LoadAbout;
import nemosofts.voxradio.fragment.FragmentDashBoard;
import nemosofts.voxradio.fragment.online.FragmentCategories;
import nemosofts.voxradio.fragment.online.FragmentCountries;
import nemosofts.voxradio.fragment.online.FragmentLatest;
import nemosofts.voxradio.fragment.online.FragmentPodcasts;
import nemosofts.voxradio.fragment.online.FragmentRecent;
import nemosofts.voxradio.fragment.online.FragmentTrending;
import nemosofts.voxradio.interfaces.AboutListener;
import nemosofts.voxradio.utils.AudioRecorder;
import nemosofts.voxradio.utils.IfSupported;
import nemosofts.voxradio.utils.advertising.AdManagerInterAdmob;
import nemosofts.voxradio.utils.advertising.AdManagerInterApplovin;
import nemosofts.voxradio.utils.advertising.AdManagerInterStartApp;
import nemosofts.voxradio.utils.advertising.AdManagerInterUnity;
import nemosofts.voxradio.utils.advertising.AdManagerInterYandex;
import nemosofts.voxradio.utils.advertising.GDPRChecker;
import nemosofts.voxradio.utils.advertising.RewardAdAdmob;
import nemosofts.voxradio.utils.advertising.RewardAdApplovin;
import nemosofts.voxradio.utils.advertising.RewardAdStartApp;
import nemosofts.voxradio.utils.advertising.RewardAdUnity;
import nemosofts.voxradio.utils.helper.Helper;

public class MainActivity extends NSoftsPlayerActivity implements NavigationView.OnNavigationItemSelectedListener {

    FragmentManager fm;
    MenuItem menuLogin;
    MenuItem menuProfile;
    ReviewManager manager;
    ReviewInfo reviewInfo;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        FrameLayout contentFrameLayout = findViewById(R.id.content_frame);
        getLayoutInflater().inflate(R.layout.content_main, contentFrameLayout);

        IfSupported.isRTL(this);
        IfSupported.isScreenshot(this);

        Callback.setIsAppOpen(true);
        helper = new Helper(this);

        fm = getSupportFragmentManager();

        navigationView.setNavigationItemSelectedListener(this);

        Menu menu = navigationView.getMenu();
        menuLogin = menu.findItem(R.id.nav_login);
        menuProfile = menu.findItem(R.id.nav_profile);

        new GDPRChecker(MainActivity.this).check();
        changeLoginName();
        loadAboutData();

        manager = ReviewManagerFactory.create(this);
        Task<ReviewInfo> request = manager.requestReviewFlow();
        request.addOnCompleteListener(task -> {
            if (task.isSuccessful()) {
                reviewInfo = task.getResult();
            }
        });

        setNavMenu();
        loadDashboardFrag();

        // requestPermission will show the native Android notification permission prompt.
        // NOTE: It's recommended to use a OneSignal In-App Message to prompt instead.
        OneSignal.getNotifications().requestPermission(false, Continue.none());

        OnBackPressedCallback callback = new OnBackPressedCallback(true) {
            @Override
            public void handleOnBackPressed() {
                handleOnBack();
            }
        };
        getOnBackPressedDispatcher().addCallback(this, callback);
    }

    private void setNavMenu() {
        navHome.setOnClickListener(view -> {
            if (!navHome.isActive()){
                FragmentDashBoard home = new FragmentDashBoard();
                loadFrag(home, getString(R.string.dashboard), fm);
            }
            bottomNavigationView(0);
        });
        navLatest.setOnClickListener(view -> {
            if (!navLatest.isActive()){
                FragmentLatest latest = new FragmentLatest();
                loadFrag(latest, getString(R.string.latest), fm);
            }
            bottomNavigationView(1);
        });
        navMost.setOnClickListener(view -> {
            if (!navMost.isActive()){
                FragmentTrending most = new FragmentTrending();
                loadFrag(most, getString(R.string.trending), fm);
            }
            bottomNavigationView(2);
        });
        navCategory.setOnClickListener(view -> {
            if (!navCategory.isActive()){
                FragmentCategories category = new FragmentCategories();
                loadFrag(category, getString(R.string.categories), fm);
            }
            bottomNavigationView(3);
        });
        navRestore.setOnClickListener(view -> {
            if (!navRestore.isActive()){
                FragmentRecent recent = new FragmentRecent();
                loadFrag(recent, getString(R.string.recently), fm);
            }
            bottomNavigationView(4);
        });
    }

    private void loadDashboardFrag() {
        FragmentDashBoard f1 = new FragmentDashBoard();
        loadFrag(f1, getResources().getString(R.string.dashboard), fm);
        navigationView.setCheckedItem(R.id.nav_home);
    }

    public void loadFrag(Fragment f1, String name, FragmentManager fm) {
        for (int i = 0; i < fm.getBackStackEntryCount(); ++i) {
            fm.popBackStackImmediate();
        }

        FragmentTransaction ft = fm.beginTransaction();
        ft.setTransition(FragmentTransaction.TRANSIT_FRAGMENT_OPEN);
        if (!name.equals(getString(R.string.dashboard))) {
            ft.hide(fm.getFragments().get(fm.getBackStackEntryCount()));
            ft.add(R.id.fragment, f1, name);
            ft.addToBackStack(name);
        } else {
            ft.replace(R.id.fragment, f1, name);
        }
        ft.commit();

        if (getSupportActionBar() != null){
            getSupportActionBar().setTitle(name);
        }

        if (mLayout.getPanelState() == SlidingUpPanelLayout.PanelState.EXPANDED) {
            mLayout.setPanelState(SlidingUpPanelLayout.PanelState.COLLAPSED);
        }
    }

    private void changeLoginName() {
        if (menuLogin != null) {
            if (spHelper.isLogged()) {
                menuProfile.setVisible(true);
                menuLogin.setTitle(getResources().getString(R.string.logout));
                menuLogin.setIcon(ContextCompat.getDrawable(this, R.drawable.ic_logout));
            } else {
                menuProfile.setVisible(false);
                menuLogin.setTitle(getResources().getString(R.string.login));
                menuLogin.setIcon(ContextCompat.getDrawable(this,R.drawable.ic_login));
            }
        }
    }

    public void loadAboutData() {
        if (NetworkUtils.isConnected(this)) {
            LoadAbout loadAbout = new LoadAbout(MainActivity.this, new AboutListener() {
                @Override
                public void onStart() {
                    // this method is empty
                }

                @Override
                public void onEnd(String success, String verifyStatus, String message) {
                    if (isFinishing() && !success.equals("1")) {
                        return;
                    }
                    dbHelper.addToAbout();
                    helper.initializeAds();
                    initAds();
                }
            });
            loadAbout.execute();
        } else {
            try {
                dbHelper.getAbout();
            } catch (Exception e) {
                Log.e("MainActivity", "Error getAbout", e);
            }
        }
    }

    private void initAds() {
        if (Boolean.TRUE.equals(Callback.getIsInterAd()) && (!spHelper.getIsSubscribed() || spHelper.getIsAdOn())) {
            switch (Callback.getAdNetwork()) {
                case Callback.AD_TYPE_ADMOB :
                    AdManagerInterAdmob adManagerInterAdmob = new AdManagerInterAdmob(getApplicationContext());
                    adManagerInterAdmob.createAd();
                    break;
                case Callback.AD_TYPE_STARTAPP :
                    AdManagerInterStartApp adManagerInterStartApp = new AdManagerInterStartApp(getApplicationContext());
                    adManagerInterStartApp.createAd();
                    break;
                case Callback.AD_TYPE_APPLOVIN :
                    AdManagerInterApplovin adManagerInterApplovin = new AdManagerInterApplovin(MainActivity.this);
                    adManagerInterApplovin.createAd();
                    break;
                case Callback.AD_TYPE_YANDEX :
                    AdManagerInterYandex adManagerInterYandex = new AdManagerInterYandex(MainActivity.this);
                    adManagerInterYandex.createAd();
                    break;
                case Callback.AD_TYPE_UNITY :
                    AdManagerInterUnity adManagerInterUnity = new AdManagerInterUnity();
                    adManagerInterUnity.createAd();
                    break;
                default:
                    break;
            }
        }
        if (Boolean.TRUE.equals(Callback.getIsRewardAd()) && (!spHelper.getIsSubscribed() || spHelper.getIsAdOn())) {
            switch (Callback.getAdNetwork()) {
                case Callback.AD_TYPE_ADMOB :
                    RewardAdAdmob rewardAdAdmob = new RewardAdAdmob(getApplicationContext());
                    rewardAdAdmob.createAd();
                    break;
                case Callback.AD_TYPE_STARTAPP :
                    RewardAdStartApp rewardAdStartApp = new RewardAdStartApp(getApplicationContext());
                    rewardAdStartApp.createAd();
                    break;
                case Callback.AD_TYPE_APPLOVIN :
                    RewardAdApplovin rewardAdApplovin = new RewardAdApplovin(MainActivity.this);
                    rewardAdApplovin.createAd();
                    break;
                case Callback.AD_TYPE_UNITY :
                    RewardAdUnity rewardAdUnity = new RewardAdUnity();
                    rewardAdUnity.createAd();
                    break;
                default:
                    break;
            }
        }
    }

    @Override
    public boolean onNavigationItemSelected(@NonNull MenuItem item) {
        int id = item.getItemId();
        if (id == R.id.nav_home){
            activateFragment(navHome, new FragmentDashBoard(), R.string.dashboard, 0);
        } else if (id == R.id.nav_latest){
            activateFragment(navLatest, new FragmentLatest(), R.string.latest, 1);
        } else if (id == R.id.nav_trending){
            activateFragment(navMost, new FragmentTrending(), R.string.trending, 2);
        } else if (id == R.id.nav_categories){
            activateFragment(navCategory, new FragmentCategories(), R.string.categories, 3);
        } else if (id == R.id.nav_recently){
            activateFragment(navRestore, new FragmentRecent(), R.string.recently, 4);
        } else if (id == R.id.nav_countries){
            checkAndLoadFragment(R.string.countries, new FragmentCountries());
        } else if (id == R.id.nav_podcasts){
            checkAndLoadFragment(R.string.podcasts, new FragmentPodcasts());
        } else if (id == R.id.nav_fav){
            openFavouriteActivity();
        } else if (id == R.id.nav_rec){
            openRecorderActivity();
        } else if (id == R.id.nav_suggest){
            openSuggestionActivity();
        } else if (id == R.id.nav_profile){
            startActivity(new Intent(MainActivity.this, ProfileActivity.class));
        } else if (id == R.id.nav_settings){
            startActivity(new Intent(MainActivity.this, SettingActivity.class));
        } else if (id == R.id.nav_login){
            helper.clickLogin();
        }
        DrawerLayout drawer = findViewById(R.id.drawer_layout);
        drawer.closeDrawer(GravityCompat.START);
        return true;
    }

    private void activateFragment(ToggleView helper, Fragment fragment, int titleResId, int bottomNavIndex) {
        if (!helper.isActive()) {
            loadFrag(fragment, getString(titleResId), fm);
        }
        bottomNavigationView(bottomNavIndex);
    }

    private void checkAndLoadFragment(int titleResId, Fragment fragment) {
        String currentTitle = String.valueOf(Objects.requireNonNull(getSupportActionBar()).getTitle());
        if (!currentTitle.equals(getString(titleResId))) {
            loadFrag(fragment, getString(titleResId), fm);
        }
        bottomNavigationView(5);
    }

    private void openFavouriteActivity() {
        if (spHelper.isLogged()) {
            Intent intentFav = new Intent(MainActivity.this, RadioByActivity.class);
            intentFav.putExtra("type", getString(R.string.favourite));
            intentFav.putExtra("id", "");
            intentFav.putExtra("name", getString(R.string.favourite));
            startActivity(intentFav);
        } else {
            helper.clickLogin();
        }
    }

    private void openRecorderActivity() {
        if (checkPermissionM()) {
            if (Callback.isRecording()) {
                Toast.makeText(this, getResources().getString(R.string.stop_record_first), Toast.LENGTH_SHORT).show();
            } else {
                startActivity(new Intent(MainActivity.this, RecorderActivity.class));
            }
        }
    }

    private void openSuggestionActivity() {
        if (spHelper.isLogged()){
            startActivity(new Intent(MainActivity.this, SuggestionActivity.class));
        } else {
            helper.clickLogin();
        }
    }

    public void bottomNavigationView(int pos) {
        if (navHome == null || navLatest == null || navMost == null || navCategory == null || navRestore == null){
            return;
        }

        // List of navigation items
        ToggleView[] navItems = {navHome, navLatest, navMost, navCategory, navRestore};

        // Special handling for pos == 5
        if (pos == 5) {
            deactivateAll(navItems);
            return;
        }

        for (int i = 0; i < navItems.length; i++) {
            if (i == pos) {
                if (!navItems[i].isActive()) {
                    navItems[i].activate();
                    navItems[i].setBadgeText("");
                }
            } else {
                if (navItems[i].isActive()) {
                    navItems[i].deactivate();
                    navItems[i].setBadgeText(null);
                }
            }
        }
    }

    private void deactivateAll(ToggleView[] navItems) {
        if (navItems == null){
            return;
        }
        for (ToggleView navItem : navItems) {
            if (navItem.isActive()) {
                navItem.deactivate();
                navItem.setBadgeText(null);
            }
        }
    }

    @Override
    public void onResume() {
        changeLoginName();
        if (Callback.isRecreate()) {
            Callback.setRecreate(false);
            recreate();
        }
        super.onResume();
    }

    @OptIn(markerClass = UnstableApi.class)
    @Override
    protected void onDestroy() {
        Callback.setIsAppOpen(false);
        if (PlayerService.getExoPlayer() != null && !PlayerService.getExoPlayer().getPlayWhenReady()) {
            Intent intent = new Intent(getApplicationContext(), PlayerService.class);
            intent.setAction(PlayerService.ACTION_STOP);
            startService(intent);
        }
        AudioRecorder.stopRecording();
        try {
            dbHelper.close();
        } catch (Exception e) {
            Log.e("MainActivity", "Error in closing", e);
        }
        super.onDestroy();
    }

    @Override
    public boolean onOptionsItemSelected(@NonNull MenuItem item) {
        if (item.getItemId() == android.R.id.home) {
            handleOnBack();
            return true;
        }
        return super.onOptionsItemSelected(item);
    }

    private void handleOnBack() {
        DrawerLayout drawer = findViewById(R.id.drawer_layout);
        if (drawer.isDrawerOpen(GravityCompat.START)) {
            drawer.closeDrawer(GravityCompat.START);
        } else if (mLayout.getPanelState().equals(SlidingUpPanelLayout.PanelState.EXPANDED)) {
            mLayout.setPanelState(SlidingUpPanelLayout.PanelState.COLLAPSED);
        } else if (fm.getBackStackEntryCount() != 0) {
            String title = fm.getFragments().get(fm.getBackStackEntryCount() - 1).getTag();
            if (title != null) {

                // Custom class to hold both navigation ID and bottom navigation index
                class NavInfo {
                    final int navId;
                    final int bottomNavIndex;
                    NavInfo(int navId, int bottomNavIndex) {
                        this.navId = navId;
                        this.bottomNavIndex = bottomNavIndex;
                    }
                }

                // Map to hold titles and corresponding NavInfo
                Map<String, NavInfo> titleToNavInfoMap = new HashMap<>();

                // Initialize the map with titles and corresponding actions
                titleToNavInfoMap.put(getString(R.string.dashboard), new NavInfo(R.id.nav_home, 0));
                titleToNavInfoMap.put(getString(R.string.nav_home), new NavInfo(R.id.nav_home, 0));
                titleToNavInfoMap.put(getString(R.string.latest), new NavInfo(R.id.nav_latest, 1));
                titleToNavInfoMap.put(getString(R.string.trending), new NavInfo(R.id.nav_trending, 2));
                titleToNavInfoMap.put(getString(R.string.categories), new NavInfo(R.id.nav_categories, 3));
                titleToNavInfoMap.put(getString(R.string.recently), new NavInfo(R.id.nav_recently, 4));
                titleToNavInfoMap.put(getString(R.string.countries), new NavInfo(R.id.nav_countries, 5));
                titleToNavInfoMap.put(getString(R.string.podcasts), new NavInfo(R.id.nav_podcasts, 5));
                titleToNavInfoMap.put(getString(R.string.search), new NavInfo(R.id.nav_home, 5));

                // Update the navigation view and bottom navigation view if the title is in the map
                NavInfo navInfo = titleToNavInfoMap.get(title);
                if (navInfo != null) {
                    navigationView.setCheckedItem(navInfo.navId);
                    bottomNavigationView(navInfo.bottomNavIndex);
                    pageChange(navInfo.bottomNavIndex);
                }
            }
            if (getSupportActionBar() != null) {
                getSupportActionBar().setTitle(Objects.equals(title, getString(R.string.dashboard)) ? getString(R.string.nav_home) : title);
            }
        } else if (reviewInfo != null){
            Task<Void> flow = manager.launchReviewFlow(MainActivity.this, reviewInfo);
            flow.addOnCompleteListener(task1 -> DialogUtil.exitDialog(MainActivity.this));
        } else {
            DialogUtil.exitDialog(MainActivity.this);
        }
    }

    private void pageChange(int bottomNavIndex) {
        if (bottomNavIndex == 0){
            FragmentDashBoard home = new FragmentDashBoard();
            loadFrag(home, getString(R.string.dashboard), fm);
        } else if (bottomNavIndex == 1){
            FragmentLatest latest = new FragmentLatest();
            loadFrag(latest, getString(R.string.latest), fm);
        } else if (bottomNavIndex == 2){
            FragmentTrending most = new FragmentTrending();
            loadFrag(most, getString(R.string.trending), fm);
        } else if (bottomNavIndex == 3){
            FragmentCategories category = new FragmentCategories();
            loadFrag(category, getString(R.string.categories), fm);
        } else if (bottomNavIndex == 4){
            FragmentRecent recent = new FragmentRecent();
            loadFrag(recent, getString(R.string.recently), fm);
        }
    }

    @NonNull
    private Boolean checkPermissionM() {
        String permission;
        if (Build.VERSION.SDK_INT >= 33) {
            permission = READ_MEDIA_AUDIO;
        } else if (Build.VERSION.SDK_INT >= 29) {
            permission = READ_EXTERNAL_STORAGE;
        } else {
            permission = WRITE_EXTERNAL_STORAGE;
        }
        if (ContextCompat.checkSelfPermission(MainActivity.this, permission) != PackageManager.PERMISSION_GRANTED) {
            permissionLauncher.launch(permission);  // Request permission using the new API
            return false;
        }
        return true;
    }

    private final ActivityResultLauncher<String> permissionLauncher = registerForActivityResult(
            new ActivityResultContracts.RequestPermission(), isGranted ->
                    Toast.makeText(MainActivity.this, Boolean.TRUE.equals(isGranted) ? "Permission granted"
                            : getResources().getString(R.string.err_cannot_use_features), Toast.LENGTH_SHORT).show()
    );
}